/*******************************************************************************************************************************
Main.c

Version 1.0   Nov 2009
Version 1.1   Dec 2009 - Changed Kmh to kmh
Version 1.2   Jun 2010 - Enabled imperial and metric display via config screens
Version 1.3   Nov 2010 - Enabled the power up timer and brownup feature to combat an intermittent power up issue
Version 1.4   Dec 2010 - Rearranged startup sequence in InitEverything() to minimise startup issues

Go to http://geoffg.net/gpscomputer.html for updates, errata and helpfull notes
    
    Copyright 2009 Geoff Graham - http://geoffg.net
    This program is free software: you can redistribute it and/or modify it under the terms of the GNU General
    Public License as published by the Free Software Foundation, either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
    implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.  You should have received a copy of the GNU General Public License along with this program.
    If not, see <http://www.gnu.org/licenses/>.


This is the main source file for the GPS Car Computer project.  The other files are:
    CCS-18F4550.h           Defines specific to the configuration of the 18F4550
    SG122232A_Driver_2_0.h  Driver for the SG122232A graphics LCD
    bootloader.h            The bootloader code in compiled form
    glcd_library_1_0a.h     Graphics library (fonts and drawing routines)
    
    Two other files are required.  These configure the USB protocol stack for this project.  Because of licensing
    restrictions these are not included in this source distribution.  Instructions for creating them are listed
    in the file "Modifying CCS USB driver to suit the GPS Car Computer.pdf"

Development Environment
    To compile this you need:
     - CCS C Compiler Version 4.0.057 or higher (www.ccsinfo.com)
     - Microchip MPLAB IDE Version 8.14 or higher (www.microchip.com)

Program summary:
    The program consists of five main sections:
    
    Timer 0 interrupt:      This occurs every 85.3uS and is used count down the timeout used to detect a faulty gps 
    module, sensing/debounce for the three push button switches, control of the backlight brightness and control of 
    the beeper used to signal overspeed.
    
    UART Receive Interrupt: Occurs on every character received from the GPS.  This routine echoes the received character 
    to the USB virtual serial port then processes it.  This involves stripping the leading $ and stripping and checking
    the checksum.  The received string is stored in GpsBuf[2][] and GpsDataReady is updated (0=no data, 1=buffer1, 2=buffer2).
    While the main program process this data the receive interrupt will be filling the second buffer.  As a message from 
    the GPS takes about 160mS the main program must finish in less than that time and reset GpsDataReady to zero otherwise 
    the next message will be lost.
    
    USB Interrupt and code:     The interrupt occurs when processing of a send or receive is required.  USB processing 
    occurs entirly in the background and does not affect the rest of the program.
    
    Main Program Loop:      This runs continuously.  It checks for received data from the GPS and processes it.  It
    also runs a state machine that switches execution to various functions depending on the value of the current
    state.  Each state represents a different display that the user sees and the function called by the state machine
    is responsible for displaying the data and processing any pertinant button presses.  The state will be switched to a 
    different state if the user presses a button that calls up a new display.
    
    Graphics and LCD driver:    Drawing on the LCD is controlled by glcd_library_1_0a.h which includes lines, circles
    and three fonts.  These routines call SG122232A_Driver_2_0.h which maintains an image of the LCD in a buffer in RAM.
    When instructed the driver will dump the buffer into the LCD resulting in a very fast update.
    
    
********************************************************************************************************************************/


// define the version number shown on the startup screen
#define VERSION         "1.4"

// comment out the following line if you do not want the USB serial emulator
#define USE_USB         1                                   

// the following controls the build and its use with the bootloader
//    USE_BOOTLOADER = 0    Will not include the bootloader and will run in low memory as normal.  Use a programmer to load.
//    USE_BOOTLOADER = 1    Will include the bootloader.  The result is a single HEX file for use with a programmer
//    USE_BOOTLOADER = 2    Will compile an update. This can only be loaded into flash by the bootloader.
#define USE_BOOTLOADER  1


/*******************************************************************************************************************************/

#include <18F4550.h>
#device ICD=TRUE
#device adc=8
#use delay(clock=48000000)

#include "CCS-18F4550.h"

#FUSES HSPLL                    //High speed Osc (> 4mhz for PCM/PCH) (>10mhz for PCD)
#FUSES NOWDT                    //No Watch Dog Timer
#FUSES NOPROTECT                //Code not protected from reading
#FUSES BROWNOUT               	//Brownout reset enabled
#FUSES BORV43					//Brownout at 4.3V
#FUSES PUT                    	//Power Up Timer enabled
#FUSES NOCPD                    //No EE protection
#FUSES STVREN                   //Stack full/underflow will cause reset
#FUSES DEBUG                    //Debug mode for ICD
#FUSES NOLVP                    //No low voltage prgming, B3(PIC16) or B5(PIC18) used for I/O
#FUSES NOWRT                    //Program memory not write protected
#FUSES NOWRTD                   //Data EEPROM not write protected
#FUSES NOIESO                   //Internal External Switch Over mode disabled
#FUSES NOFCMEN                  //Fail-safe clock monitor disabled
#FUSES NOPBADEN                 //PORTB pins are configured as analog input channels on RESET
#FUSES NOWRTC                   //configuration not registers write protected
#FUSES NOWRTB                   //Boot block not write protected
#FUSES NOEBTR                   //Memory not protected from table reads
#FUSES NOEBTRB                  //Boot block not protected from table reads
#FUSES NOCPB                    //No Boot Block code protection
#FUSES MCLR                     //Master Clear pin enabled
#FUSES NOXINST                  //Extended set extension and Indexed Addressing mode disabled (Legacy mode)
#FUSES PLL5                     //Divide By 5(20MHz oscillator input)
#FUSES CPUDIV1                  //System Clock by 1
#FUSES USBDIV                   //USB clock source comes from PLL divide by 2
#FUSES VREGEN                   //USB voltage regulator disabled
#FUSES NOICPRT                  //ICPRT disabled


/************************************************************************************************
Define the i/o pins
************************************************************************************************/
#define     TRISA_INIT          0b00001101  // this MUST agree with the #defines below
#define     CMP_IN_NEG          RA0     // input analog - Comparator negative
#define     LCD_A0              RA1     // output - LCD A0
#define     REF_OUT             RA2     // output analog - reference voltage
#define     CMP_IN_POS          RA3     // input analog - Comparator positive
#define     CMP_OUT             RA4     // output - Comparator output
#define     BEEPER              RA5     // output - the beeper used to signal overspeed

#define     TRISB_INIT          0b11111000  // this MUST agree with the #defines below
                                        // pullups available (all or nothing)
#define     LCD_RES             RB0     // output - LCD reset
#define     LCD_E               RB1     // output - LCD enable (data strobe)
#define     LCD_RW              RB2     // output - LCD Read/Write
#define     SET_BTN             RB3     // input - the set button
#define     UP_BTN              RB4     // input - the up button
#define     DWN_BTN             RB5     // input - the down button
#define     CAR_FUEL            RB6     // input - car's fuel injector solenoid -also- ICD PGD
#define     CAR_LIGHTS          RB7     // input - wired to the car's headlamps -also- ICD PGC

#define     TRISC_INIT          0b10001000  // this MUST agree with the #defines below
#define     LCD_CS1             RC0     // output - LCD Chip Select 1
#define     LCD_CLK             RC1     // output - LCD clock used to drive the glass
#define     LCD_BKLIT           RC2     // output - LCD backlight
#define     UNAVAILABLE         RC3     // Unavailable on 18F2550 and 18F4550
#define     USB_DN              RC4     // output - USB D-
#define     USB_DP              RC5     // output - USB D+
#define     GPS_TX              RC6     // output - Data to GPS
#define     GPS_RX              RC7     // input - GPS data

#define     TRISD_INIT          0b00000000  // this MUST agree with the #defines below
#define     D0                  RD0     // input & output - Data bus to LCD
#define     D1                  RD1     // input & output - Data bus to LCD
#define     D2                  RD2     // input & output - Data bus to LCD
#define     D3                  RD3     // input & output - Data bus to LCD
#define     D4                  RD4     // input & output - Data bus to LCD
#define     D5                  RD5     // input & output - Data bus to LCD
#define     D6                  RD6     // input & output - Data bus to LCD
#define     D7                  RD7     // input & output - Data bus to LCD

#define     TRISE_INIT          0b00000000  // this MUST agree with the #defines below
#define     LCD_CS2             RE0     // output - LCD Chip Select 2
#define     UNASSIGNED1         RE1     // output - for future expansion
#define     UNASSIGNED2         RE2     // output - for future expansion

#define     LCD_DATA            PORTD           // Data port for the LCD
#define     LCD_DATA_TRIS       TRISD           // and its associated direction register    


// define debugging signals - not used in production code
#define Mark1               { UNASSIGNED1 = 1; UNASSIGNED1 = 0; }
#define Mark1double         { UNASSIGNED1 = 1; UNASSIGNED1 = 0; UNASSIGNED1 = 1; UNASSIGNED1 = 0; }
#define Mark2               { UNASSIGNED2 = 1; UNASSIGNED2 = 0; }
#define Mark2double         { UNASSIGNED2 = 1; UNASSIGNED2 = 0; UNASSIGNED2 = 1; UNASSIGNED2 = 0; }
#define Mark3               { UNASSIGNED3 = 1; UNASSIGNED3 = 0; }
#define Mark3double         { UNASSIGNED2 = 1; UNASSIGNED2 = 0; UNASSIGNED2 = 1; UNASSIGNED2 = 0; }


/************************************************************************************************
Define the various states that the display can be in.  This is part of the main State Machine

To add a new display screen you must add a state:
    1.  Insert the new state label in enum StateValues
    2.  Insert the actions for this state in the corresponding position in the array StateAction
    3.  If it is a show/hideable state:
            a.  Insert the descriptive string for the new state in M_set_showlist
            b.  add an entry to the show/hide table in the eeprom configuration data
            c.  Add one to the value of HIDE_SIZE
    4.  Write a subrouting to process the new state
    5.  Insert the subroutine call in the switch statement in main()
    
************************************************************************************************/

// Values used by the state machine
// This lists all the states that the display can be in
enum StateValues {
    S_Time,
    S_Speed,
    S_Fuel,
    S_Dist,
    S_Alt,
    S_Posit,
    S_Sat,                              // must be the last normal screen (reachable by UP/DOWN buttons)
    
    S_TimeCfg,
    S_SpeedUnitsCfg,
    S_SpeedCfg,
    S_FuelCfg,
    S_DistCfg,
    S_AltUnitsCfg,
    S_AltCfg,
    S_BrightDayCfg,
    S_BrightNightCfg,
    S_LAST_ENTRY
} State;

// State Change Table
// this lists the reaction of the state to various button pushes
// this is in EXACTLY the same sequence as the enum table above
// each entry specifies the state to change to when the button is pressed
// entry 1 is the Set button, 2 is the Up button and 3 is the Down button
// using the current state means that nothing will happen
const uint8 StateAction[S_LAST_ENTRY][3] = {
    S_TimeCfg, S_Sat, S_Speed,                                  // S_Time state
    S_SpeedUnitsCfg, S_Time, S_Fuel,                            // S_Speed state
    S_FuelCfg, S_Speed, S_Dist,                                 // S_Fuel state
    S_DistCfg, S_Fuel, S_Alt,                                   // S_Dist state
    S_AltUnitsCfg, S_Dist, S_Posit,                             // S_Alt state
    S_BrightDayCfg, S_Alt, S_Sat,                               // S_Posit state
    S_BrightDayCfg, S_Posit, S_Time,                            // S_Sat state - this MUST be the last normal state
    ///////////////////    place all normal screens (reachable by UP/DOWN buttons) before here
    S_Time, S_TimeCfg, S_TimeCfg,                               // S_TimeCfg state
    S_SpeedCfg, S_SpeedUnitsCfg, S_SpeedUnitsCfg,               // S_SpeedUnitsCfg state
    S_Speed, S_SpeedCfg, S_SpeedCfg,                            // S_SpeedCfg state
    S_Fuel, S_FuelCfg, S_FuelCfg,                               // S_FuelCfg state
    S_Dist, S_DistCfg, S_DistCfg,                               // S_DistCfg state
    S_AltCfg, S_AltUnitsCfg, S_AltUnitsCfg,                     // S_AltUnitsCfg state
    S_Alt, S_AltCfg, S_AltCfg,                                  // S_AltCfg state
    S_BrightNightCfg, S_BrightDayCfg, S_BrightDayCfg,           // S_BrightDayCfg state
    S_Sat, S_BrightNightCfg, S_BrightNightCfg                   // S_BrightNightCfg state
};


/************************************************************************************************
Configuration defines
************************************************************************************************/
#define GPS_TIMEOUT         255                                 // each unit is 26.3mS
#define GPS_BUFF_SIZE       85                                  // two buffers of this size are created                                         
#define LOW_SIGNAL_HOLD     10                                  // wait this many messages before removing low sig msg
#define KM_MIN_SIZE         10                                  // size of the array used to determine arrival time
#define BEEP_ON             150                                 // length of the two overspeed beeps in mS
#define BEEP_GAP            200                                 // gap between the two overspeed beeps in mS

// "factory default" values used on initial programming and when reset all is selected
#define D_state             S_Speed                             // state
#define D_ospeed            110                                 // overspeed alarm value
#define D_tz                80                                  // timezone in hours * 10
#define D_dist_hi           0                                   // distance to destination, high byte
#define D_dist_lo           0                                   // distance to destination, low byte
#define D_bright_day        100                                 // day brightness
#define D_bright_night      20                                  // night brightness
#define D_fuel              25                                  // default multiplier for the fuel display
#define D_pointer           1                                   // compass pointer, 0=north, 1=heading
#define D_AutoScan          0                                   // Auto scan, 0=off, 2=running
#define D_speedunits		0									// Default speed units is kmh
#define D_altitudeunits		0									// Default altitude units is metres
#define D_hide_clock        1                                   // show/hide for the clock
#define D_hide_speed        0                                   // show/hide for the speed
#define D_hide_economy      0                                   // show/hide for economy
#define D_hide_destination  0                                   // show/hide for destination
#define D_hide_compass      0                                   // show/hide for the compass
#define D_hide_latlong      1                                   // show/hide for latlong
#define D_hide_signal       1                                   // show/hide for the signal



/************************************************************************************************
Implement the build options (see #define USE_BOOTLOADER  in the beginning of this file)
************************************************************************************************/
#if USE_BOOTLOADER == 1
    #build(reset=0x800, interrupt=0x808)
    #org 0, 0x7ff 
    void bootloader_code(void) {
        #asm
            goto 0x6FC              // this is a "magic" address found by disassembly of the bootloader
            nop
            nop
            goto 0x808
        #endasm
    }
    #include "bootloader.h"         // this is the bootloader (in object form)
#endif

#if USE_BOOTLOADER == 2
    #build(reset=0x800, interrupt=0x808)
    #org 0, 0x7ff {}
#endif

#include "SG12232A_Driver_2_0.h"
#include "glcd_library_1_0a.h"

#ifdef USE_USB
    #include "usb_cdc.h"
#endif


/************************************************************************************************
Global memory locations
************************************************************************************************/
bit NewData;                                                // true when new data is available for display
uint8 GpsTimeout;                                           // used to count down the timeout for gps data

bit SetBtnEvent;                                            // true if Set button pressed
uint8 UpBtn, UpBtnEvent;                                    // count of the number of Up button presses
uint8 DwnBtn, DwnBtnEvent;                                  // count of the number of Down button presses
uint8 AutoScan;                                             // 0=off, 1=waiting for both buttons to release, 2=auto scan running
uint8 SecCnt;                                               // counts seconds in the main loop for timing Auto Scan

uint8 StrBuf[18];

bit NoSignal, Startup;
bit BkLightTime;                                            // true if backlight is controlled by the time
uint8 BkLightValue;                                         // used to control the duty cycle of the backlight

uint8 hour, min, sec;                                       // time from the gps (utc)
sint8 tz;                                                   // timezone in hours*10
sint16 minutes;                                             // local time
uint8 h, m;                                                 // local time
bit pm;                                                     // local time
    
sint32 KmSecRemaining;
sint16 KmMin[KM_MIN_SIZE];
sint16 FuelCnt1;                                            // count of how long the fuel injector has been open
sint16 FuelCnt2;
uint8 FuelEfficiency;

uint8 LatBuf[16], LongBuf[16];
uint8 Speed;
bit OverSpeed;
sint16 Heading;
sint16 Altitude;
uint8 Sv[12];
uint8 SvCnt, SatUsed, FoundSatCnt;

uint8 GpsBuf[2][GPS_BUFF_SIZE];
uint8 GpsDataReady;


/************************************************************************************************
Declare functions
************************************************************************************************/
void InitEverything(void);
void GetGPSData(void);
void GPS_SetupUART(void);
void DoSpeed(void);
void DoSpeedUnitsCfg(void);
void DoSpeedCfg(void);
void DoMsg(void);
void DoKmRemaining(void);
void CalcTime(void);
void DoTime(void);
void DoTimeCfg(void);
void DoFuel(void);
void DoFuelCfg(void);
void DoPosit(void);
void DoDist(void);
void DoAlt(void);
void DoAltUnitsCfg(void);
void DoDistCfg(void);
void DoAltCfg(void);
void DrawSat(void);
void DoSat(void);
void DoBrightDayCfg(void);
void DoBrightNightCfg(void);
void ExtractGpsData(void);
uint16 GetGpsInt(uint8 *p, uint8 i);
uint16 GetGpsDecimal(uint8 *p, uint8 i);
void GetTime(uint8 *p);
void GetLatLong(uint8 *p, uint8 *b);
void Center812(uint8 *p, uint8 x, uint8 y);
void Center1116(uint8 *p, uint8 x, uint8 y);
void SetShowHide(void);
void ResetAll(void);
void CopySpeedUnits(void);
void CopyDistanceUnits(void);


/************************************************************************************************
Initialise the eeprom configuration data.
************************************************************************************************/
#define E_state             0                               // index values of the stored configuration data
#define E_ospeed            1
#define E_tz                2
#define E_dist_hi           3
#define E_dist_lo           4
#define E_bright_day        5
#define E_bright_night      6
#define E_fuel              7
#define E_pointer           8
#define E_AutoScan          9
#define E_speedunits		10
#define E_altitudeunits		11

#define HIDE_START          0x10                            // start of the show/hide array in EEPROM
#define HIDE_SIZE           7                               // size of the show/hide array
#define E_hide_clock        HIDE_START
#define E_hide_speed        HIDE_START + 1
#define E_hide_economy      HIDE_START + 2
#define E_hide_destination  HIDE_START + 3
#define E_hide_compass      HIDE_START + 4
#define E_hide_latlong      HIDE_START + 5
#define E_hide_signal       HIDE_START + 6

#define BACKUP_START        0x80


// initialise the eeprom default values for the configurable items
#ROM int8 0xF00000 = {  
                        D_state,
                        D_ospeed,
                        D_tz,
                        D_dist_hi,
                        D_dist_lo,
                        D_bright_day,
                        D_bright_night,
                        D_fuel,
                        D_pointer,
                        D_AutoScan,
                        D_speedunits,
                        D_altitudeunits
                    }   
                                        
// initialise the eeprom default values for the show/hide table
#ROM int8 0xF00000 + HIDE_START = {
                        D_hide_clock,
                        D_hide_speed,
                        D_hide_economy,
                        D_hide_destination,
                        D_hide_compass,
                        D_hide_latlong,
                        D_hide_signal
                        }

// initialise the eeprom backup values for the configurable items
#ROM int8 0xF00000 + BACKUP_START = {
                        D_state,
                        D_ospeed,
                        D_tz,
                        D_dist_hi,
                        D_dist_lo,
                        D_bright_day,
                        D_bright_night,
                        D_fuel,
                        D_pointer,
                        D_AutoScan,
                        D_speedunits,
                        D_altitudeunits
                    }   
                                        
// initialise the eeprom backup values for the show/hide table
#ROM int8 0xF00000 + BACKUP_START + HIDE_START = {
                        D_hide_clock,
                        D_hide_speed,
                        D_hide_economy,
                        D_hide_destination,
                        D_hide_compass,
                        D_hide_latlong,
                        D_hide_signal
                        }
               

/************************************************************************************************
To save space we store strings in the program flash
************************************************************************************************/
const uint8     M_SiliconChip[] = {"SILICON CHIP"};
const uint8     M_Version[]     = {"VERSION "VERSION};
//const uint8     M_Version[]     = {"TEST VER "VERSION};
const uint8     M_srch[]        = {"SEARCHING"};
const uint8     M_NoSignal[]    = {"LOW SIGNAL"};
const uint8     M_err[]         = {"ERROR"};
const uint8     M_gpsfault[]    = {"GPS FAULT"};
const uint8		M_setspeedunits[]={"SET SPEED UNITS"};
const uint8		M_setheightunits[]={"SET HEIGHT UNIT"};
const uint8     M_kmh[]         = {"kmh"};
const uint8		M_kn[]			= {"kn"};
const uint8		M_mph[]			= {"mph"};
const uint8     M_km[]          = {"km"};
const uint8		M_nm[]			= {"nm"};
const uint8		M_miles[]		= {" miles"};
const uint8		M_metres[]		= {"METRES"};
const uint8		M_feet[]		= {"FEET"};
const uint8     M_setspeed[]    = {"SET SPEED ALARM"};
const uint8     M_setdest[]     = {"SET DESTINATION"};
const uint8     M_setfuel[]     = {"DISPLAY SCALE"};
const uint8     M_remain[]      = {"REMAINING"};
const uint8     M_off[]         = {"OFF"};
const uint8     M_settime[]     = {"SET LOCAL TIME"};
const uint8     M_setpointer[]  = {"SET POINTER"};
const uint8     M_heading[]     = {"HEADING"};
const uint8     M_due_north[]   = {"DUE NORTH"};
const uint8     M_alt[]         = {"ALTITUDE"};
const uint8     M_bright_day[]  = {"DAY BACKLIGHT"};
const uint8     M_bright_night[]= {"NIGHT BACKLIGHT"};
const uint8     M_set_showlist[]= {"CLOCK\0SPEED\0ECONOMY\0DESTINATION\0COMPASS/ALT\0LAT/LONG\0SIGNAL LEVEL"};
const uint8     M_set_show[]    = {"SHOW"};
const uint8     M_set_show_auto[]={"AUTO SCAN HIDE"};
const uint8     M_set_hide[]    = {"ALWAYS HIDE"};
const uint8     M_gps_reset[]   = {"$PSRF104,00,00,00,00,00,00,12,08*29\r\n"};



/************************************************************************************************
timer0 interrupt - this occurs every 85.3uS
************************************************************************************************/
#int_timer0
void timer0_interrupt() {
    static uint8 scale = 0;
    static uint8 SetBtnCnt = 3;
    static uint8 UpBtnCnt = 0;
    static uint8 DwnBtnCnt = 0;
    static uint8 OverSpeedCnt = 0;
    static uint8 HalfSecCnt = 0;
    uint8 i;
    
    if(!CAR_FUEL) FuelCnt1++;                                       // count if the fuel injectors are open
    
    RBPU = 0;                                                       // turn pullups on while checking the buttons
    if(SET_BTN) SetBtnCnt = 3;                                      // check the set button
    if(UP_BTN) UpBtnCnt = 0;                                        // and the up button
    if(DWN_BTN) DwnBtnCnt = 0;                                      // and down
    RBPU = 1;                                                       // turn off pullups, they interfere with injector detect
    
    if(++scale == 0) {                                              // this code is excuted once every 21.8 mS
        if(SetBtnCnt == 1) SetBtnEvent = true;
        if(SetBtnCnt) SetBtnCnt--;
        
        if(UpBtnCnt++ == 2) UpBtnEvent++;                           // first step
        if(UpBtnCnt == 55) { UpBtnEvent++; UpBtnCnt = 52; }         // after 2 sec do 10 steps per sec
        if(DwnBtnCnt++ == 2) DwnBtnEvent++;                         // first step
        if(DwnBtnCnt == 55) { DwnBtnEvent++; DwnBtnCnt = 52; }      // after 2 sec do 10 steps per sec

        // this section checks if auto scan has been triggered by pressing both up and down buttons simultaneously
        // auto scan will automatically step to the next display every 3 seconds and is cancelled by pressing up or down
        // The AutoScan variable has three states.  0 = not running
        //                                          1 = waiting for both buttons to be released
        //                                          2 or greater = auto scan running
        if(State <= S_Sat) {        
            if(!UP_BTN && !DWN_BTN && (UpBtnEvent || DwnBtnEvent))
                AutoScan = 1;
                
            if(AutoScan == 1)
                if(!UP_BTN || !DWN_BTN)
                    UpBtnEvent = DwnBtnEvent  = 0;
                else {
                    SecCnt = 0;
                    AutoScan = 2;
                }   
            
            if(AutoScan >= 2 && (UpBtnEvent || DwnBtnEvent || SetBtnEvent))
                SecCnt = AutoScan = UpBtnEvent = DwnBtnEvent  = 0;
        }
        else
            AutoScan = 0;   
            
        if(GpsTimeout) 
            GpsTimeout--;
                
        // if we are overspeed start the counter.  It is used to time the two beeps and hold after
        // they have been sounded.  Reset counter if we are not overspeed ready for the next time
        if(OverSpeed)
            OverSpeedCnt++;
        else
            OverSpeedCnt = 0;

        if(OverSpeedCnt > 0 && OverSpeedCnt < BEEP_ON/22)
            BEEPER = 1;
        else if(OverSpeedCnt < BEEP_ON/22 + BEEP_GAP/22)
            BEEPER = 0;
        else if(OverSpeedCnt < BEEP_ON/11 + BEEP_GAP/22)
            BEEPER = 1;
        else if(OverSpeedCnt > BEEP_ON/11 + BEEP_GAP/22)
            { BEEPER = 0; OverSpeedCnt--; }                 // reverse the increment so the counter freezes
            
        // set up the brightness of the backlight depending on the control mode and day/night
        // this routine makes a slow adjustment (1 percentage point every half second)
        // the result is saved in BkLightValue which is used above for PWM control
        HalfSecCnt++;
        if(HalfSecCnt > 22) {                               // this executes every 500mS
            HalfSecCnt = 0;
            if((BkLightTime && ((h < 6 && pm) || (h >= 6 && !pm))) || (!BkLightTime && !CAR_LIGHTS))
                i = E_bright_day;
            else
                i = E_bright_night;
            if(eeprom_read(i) > BkLightValue) BkLightValue++;
            if(eeprom_read(i) < BkLightValue) BkLightValue--;

        }
    set_pwm1_duty(BkLightValue);                    // 0 for full off, 100 for full on
    }   
}   




/************************************************************************************************
UART receive interrupt
************************************************************************************************/
#int_rda
void rds_interrupt() {
    #define GET_1ST_CKSUM       253
    #define GET_2ND_CKSUM       254
    #define GET_WAITING         255
    
    static uint8 GpsBufSelect = 0;
    static uint8 CharCnt = GET_WAITING;
    static uint8 GpsChecksum;
    uint8 ch;
    
    // look for a framing or overrun error and clear
    if(OERR) { CREN = OFF; CREN = ON; ch = RCREG; ch = RCREG; CharCnt = GET_WAITING; }
    if(FERR) { ch = RCREG; ch = RCREG; CharCnt = GET_WAITING; }
    
    ch = (RCREG & 0x7f);                                    // get the character
    
    #ifdef USE_USB
        if(usb_enumerated()) usb_cdc_putc_fast(ch);         // send to the USB if we are connected
    #endif
    
    if(ch == 0x0d) CharCnt = GET_WAITING;
    
    if(ch == '$') 
        CharCnt = GpsChecksum = 0;
    else if(CharCnt != GET_WAITING) {
        if(ch == '*') {                                     // signals start of checksum
            GpsBuf[GpsBufSelect][CharCnt] = 0;
            CharCnt = GET_1ST_CKSUM;
        } else if(CharCnt == GET_1ST_CKSUM) {               // first checksum byte
            CharCnt = GET_2ND_CKSUM;
            ch = ch - '0';
            if(ch > 9) ch -= 'A' - 58;                      // convert from hex
            if(ch != (GpsChecksum >> 4)) CharCnt = GET_WAITING;     // and test
        } else if(CharCnt == GET_2ND_CKSUM) {               // second checksum byte
            ch = ch - '0';
            if(ch > 9) ch -= 'A' - 58;                      // convert from hex
            if(ch == (GpsChecksum & 0x0f)) {                // and test
                if(GpsDataReady == 0) {
                    GpsBufSelect++;
                    GpsDataReady = GpsBufSelect;            // success
                    if(GpsBufSelect > 1) GpsBufSelect = 0;
                }   
                GpsTimeout = GPS_TIMEOUT;
            }   
            CharCnt = GET_WAITING; 
        } else {
            GpsChecksum = GpsChecksum ^ ch;
            if(ch >= 'a') ch -= 'a' - 'A';                  // convert to uppercase
            GpsBuf[GpsBufSelect][CharCnt] = ch; CharCnt++;
            if(CharCnt >= GPS_BUFF_SIZE) CharCnt = GET_WAITING;
        }   
    }   
}   
    


/**********************************************************************************************
Main program
This first initialises everything then enters an endlass loop which consists of:
  -  Checking if a char has arrived over the USB and if so, sends it to the GPS
  -  If there is data from the GPS it decodes that data and loads it into global variables
  -  Calculates the Km and time remaining
  -  Calls the display handler for the current page on the LCD.  This and the handling of the 
     button presses is implemented in a state machine.
This loop executes at high speed (approx 5uS when there is nothing to be done)
**********************************************************************************************/
void main() {
    uint8 NewState;

    InitEverything();
    
    while(forever) {
        #ifdef USE_USB
            usb_task();
            if (usb_enumerated() && usb_cdc_kbhit() && TRMT) TXREG = usb_cdc_getc();
        #endif
        if(GpsDataReady) GetGpsData();
        if(NewData) DoKmRemaining();

        if(NewData) if(SecCnt++ == 3) SecCnt = 1;                       // used to determine when to save the state in eeprom
        
        // check if we have an error and if so, display an appropiate message
        // there are three possible errors
        //  - GPS module failure (GpsTimeout == 0)
        //  - Startup (Startup = true).  Not strictly an error but displayed while finding satellites
        //  - Low signal (NoSignal = true)
        if(GpsTimeout == 0 || Startup || NoSignal) {
            if((Startup || NoSignal) && !SET_BTN)
                DoSat();                                                // show the satellite signal screen if Set pressed
            else {
                glcd_fillScreen(0);
                
                // 
                if(GpsTimeout == 0)
                    strcpy(StrBuf, M_err);                              // timeout error (ie, gps not working)
                else if(Startup)
                    strcpy(StrBuf, M_srch);                             // starting up so we are searching
                else
                    strcpy(StrBuf, M_NoSignal);                         // low signal error
                Center812(StrBuf, 60, 0);                               // print the first line
                
                if(GpsTimeout == 0)
                    strcpy(StrBuf, M_gpsfault);
                else
                    sprintf(StrBuf, "FOUND %u SAT", FoundSatCnt);
                Center812(StrBuf, 60, 17);                              // print the second line
                
                glcd_update();
                SecCnt = 0;
            }   
            SecCnt = SetBtnEvent = UpBtnEvent = DwnBtnEvent = 0;
        } 
        // if not an error we can display the data
        else {
            // First section of the state machine
            // Refer to the StateValue and StateAction tables defined in the front of this file
            // Here we process any button presses that will cause a change of state.  To do this we look up the 
            // current state and button that has been pressed to determine what the new state is.  If the new 
            // state is hidden we repeat the process for the hidden state until we reach a non hidden state.
            do {
                if(SetBtnEvent) {                                       // Set button
                    NewState = StateAction[State][0];
                    SecCnt = 0;
                }   
                else if(UpBtnEvent) {                                   // Up button
                    NewState = StateAction[State][1];
                    UpBtn = UpBtnEvent;
                    SecCnt = 0;
                }   
                else if(DwnBtnEvent || (NewData && SecCnt == 2 && AutoScan == 2)) { // Down button or auto scan
                    NewState = StateAction[State][2];
                    if(DwnBtnEvent) SecCnt = 0;
                    DwnBtn = DwnBtnEvent;
                }
                else
                    break;
                
                // we have the new state.  Switch to it if it is NOT hidden, otherwise do the loop again for the new state
                State = NewState;
                if(State > S_Sat || (AutoScan == 0 && eeprom_read(HIDE_START + State) < 2) || (AutoScan >= 2 && eeprom_read(HIDE_START + State) < 1)) {
                    NewData = true;
                    SetBtnEvent = UpBtnEvent = DwnBtnEvent = 0;
                    break;
                }   
            } while (forever);
            
            // Second section of the state machine
            // Depending on the current state we call the appropiate function to display the data or process button presses.
            // Because the main loop executes very rapidly the function is called thousands of times a second.
            // It is up to the function to determine if it has to do anything by checking for various button presses, etc.
            switch(State) {
                case S_Time:            DoTime(); break;
                case S_TimeCfg:         DoTimeCfg(); break;
                case S_Speed:           DoSpeed(); break;
                case S_SpeedUnitsCfg:   DoSpeedUnitsCfg(); break;
                case S_SpeedCfg:        DoSpeedCfg(); break;
                case S_Fuel:            DoFuel(); break;
                case S_FuelCfg:         DoFuelCfg(); break;
                case S_Dist:            DoDist(); break;
                case S_DistCfg:         DoDistCfg(); break;
                case S_Alt:             DoAlt(); break;
                case S_AltUnitsCfg:     DoAltUnitsCfg(); break;
                case S_AltCfg:          DoAltCfg(); break;
                case S_Posit:           DoPosit(); break;
                case S_Sat:             DoSat(); break;
                case S_BrightDayCfg:    DoBrightDayCfg(); break;
                case S_BrightNightCfg:  DoBrightNightCfg(); break;
                }
            
            // Write the current state and autoscan status to eeprom.  This means that the state can be restored.
            // on power up. This code waits for 3 seconds before writing in case the change of state was caused by power off.
            if(NewData && SecCnt == 2) {
                if((State != eeprom_read(E_state)) && AutoScan == 0) eeprom_write(E_state, State);
                if(AutoScan != 1 && eeprom_read(E_AutoScan) != AutoScan) eeprom_write(E_AutoScan, AutoScan);
            }   
            NewData = UpBtn = DwnBtn = 0;                           // clear all flags
        }
    }                                                               // and continue looping forever
}
    
void InitEverything(void) {
	uint8 i;

    setup_adc_ports(NO_ANALOGS|VSS_VDD);
    setup_adc(ADC_OFF);
    setup_psp(PSP_DISABLED);
    setup_spi(SPI_SS_DISABLED);
    setup_wdt(WDT_OFF);
   
	// set up the outputs before we turn them on
    PORTE = PORTD = PORTC = PORTB = PORTA = 0;
    LCD_CS2 = LCD_CS1 = LCD_A0 = LCD_RW = 1;
    LCD_E = 0;
    LCD_RES = 0;
    LCD_DATA = 0xff;
    
    // set pin directions
    TRISA = TRISA_INIT;
    TRISB = TRISB_INIT;
    TRISC = TRISC_INIT;
    TRISD = TRISD_INIT;
    TRISE = TRISE_INIT;

    
	mSec(20);												// let everything settle
    CVRCON = 0b11100110;                                    // voltage reference ON, output on RA2, 1.35V with 5.3V Vdd
    CMCON =  0b00000001;                                    // comparator 1 ON as independent with output
    
    OverSpeed = false;
    FuelCnt2 = FuelCnt1 = 0;
    
    GpsDataReady = 0;
    Heading = 0;
    NewData = false;
    GpsTimeout = GPS_TIMEOUT;
    FoundSatCnt = SatUsed = SvCnt = 0;
    BkLightValue = 50;
    NoSignal = false;
    Startup = true;
    GPS_SetupUART();
    GpsDataReady = 0;
//    SetBtnEvent = false;
//    UpBtn = UpBtnEvent = 0;
//    DwnBtn = DwnBtnEvent = 0;
    
    // get the current state and timezone from the eeprom
    AutoScan = eeprom_read(E_AutoScan);
    State = eeprom_read(E_state);
    tz = eeprom_read(E_tz);

    // get the saved Km remaining from eeprom and zero the speed history
    KmSecRemaining = Make32(eeprom_read(E_dist_hi), eeprom_read(E_dist_lo)) * 3600;
    for(i = 0; i < KM_MIN_SIZE; i++) KmMin[i] = 0;

    setup_timer_2(T2_DIV_BY_16,99,2);                       // setup timer 2 for PWM
    setup_ccp1(CCP_PWM);                                    // enable PWM
    setup_ccp2(CCP_PWM);                                    // enable PWM
    set_pwm1_duty(100);
    set_pwm2_duty(50);
    
    LCD_RES = 1;                                            // release the LCD from reset
	mSec(100);												// let everything settle
    GLCD_Init();                                            // LCD initialise
    glcd_fillScreen(0);                                     // clear our image buffer
    
    setup_timer_0(RTCC_INTERNAL | RTCC_8_BIT | RTCC_DIV_4);
    setup_timer_1(T1_DISABLED);
    setup_timer_3(T3_DISABLED|T3_DIV_BY_1);
    
    enable_interrupts(INT_TIMER0);
    disable_interrupts(GLOBAL);
    enable_interrupts(INT_RDA);
    disable_interrupts(GLOBAL);
    
    RBPU = 0;                                               // turn on pullups
    if(!UP_BTN) SetShowHide();                              // Up button pressed, goto Show-Hide mode
    if(!DWN_BTN) {                                          // Down button pressed
        glcd_update();                                      // send a blank screen to the LCD
        ResetAll();                                         // reset everything
        while(!DWN_BTN);                                    // and wait for the button to be released
    }   
    RBPU = 1;                                               // turn off pullups
    
    // show the initial startup screen and wait 2 seconds
    strcpy(StrBuf, M_SiliconChip); Center812(StrBuf, 60, 0);
    strcpy(StrBuf, M_Version); Center812(StrBuf, 60, 17);
    glcd_update();
    mSec(2000);
    
    // clear any random button presses
    SetBtnEvent = false;
    UpBtn = UpBtnEvent = 0;
    DwnBtn = DwnBtnEvent = 0;
	
	#ifdef USE_USB
	    usb_init_cs();
	#endif
	
    enable_interrupts(GLOBAL);
}



/************************************************************************************************
Get data from the GPS module
The message from the gps module has been loaded into a buffer.  This first determines what sort
of message that it is, then extracts the various data that we want.
                    
Format of the NEMA messages that we are interested in:
    01234567890123456789012345678901234567890123456789012345678901234567890123456789
    $GPGGA,043400.000,3158.7598,S,11552.8693,E,1,05,3.4,25.0,M,-29.3,M,,0000*58
    ====== ======     =========== ============   ==     ====                 ==
    fixed   time       latitude   longtitude     sat  altitude            checksum
    header hhmmss     ddmm.mmmm   dddmm.mmmm     cnt   meters

    $GPRMC,043356.000,A,3158.7599,S,11552.8689,E,0.24,54.42,101008,,*20
    ====== ======     = =========== ============ ==== ===== ======   ==
    fixed   time   valid  latitude   longtitude  speed course date    checksum
    header hhmmss   data  ddmm.mmmm dddmm.mmmm   knots deg   ddmmyy
    
************************************************************************************************/
void GetGpsData(void) {
    uint8 *p, i;
    sint32 t;
    
    p = &GpsBuf[GpsDataReady - 1][0];
    
	// decode the GGA meassage (this gives us most of the data that we are interested in)
    if(p[2] == 'G' && p[3] == 'G' && p[4] == 'A') {
        if(GetGpsInt(p, 6) == 0)
            NoSignal = true;                                        // we don't have a good lock on enough satellites
        else {  
            GetTime(p);
            CalcTime();
            LatBuf[0] = ' ';
            GetLatLong(&p[17], &LatBuf[1]);
            LongBuf[0] = p[29];
            GetLatLong(&p[30], &LongBuf[1]);
            SatUsed = GetGpsInt(p, 7);
            Altitude = GetGpsInt(p, 9);
            if(eeprom_read(E_altitudeunits) == 1) Altitude = Altitude*3 + (Altitude/100)*28;
            if(Speed > 0) {
                // calculate the fuel used per Km averaged over 2 seconds.  Scale to suit the display (120 pixels wide)
                t = (((sint32)FuelCnt1 + (sint32)FuelCnt2) * (sint32)eeprom_read(E_fuel)) / ((sint32)Speed * 40);
                if(t > 120) t = 120;
                FuelEfficiency = make8(t, 0);
				//FuelEfficiency = 15;
            } else
                FuelEfficiency = 0;
            FuelCnt2 = FuelCnt1;                                    // 2 is the previous seconds consumption, 1 ia the current
            FuelCnt1 = 0;
        }   
        NewData = true;                                             // signal that we have new data for display

	// decode the GSV messages (this gives us the signal levels and we get three of them every four seconds)
    } else if(p[2] == 'G' && p[3] == 'S' && p[4] == 'V') {
        if(p[8] == '1') {                                           // first GSV message
            for(i = 0; i < 12; i++) Sv[i] = 0;                      // zero our array of signal levels
            SvCnt = GetGpsInt(p, 3);                                // nbr of satellites that should be in the sky
            for(i = 0; i < 4; i++) Sv[i] = GetGpsInt(p, 7 + (i * 4));   // signal levels of the first four satellites
        }   
        if(p[8] == '2') for(i = 0; i < 4; i++) Sv[i + 4] = GetGpsInt(p, 7 + (i * 4));   // second GSV message
        if(p[8] == '3') for(i =0; i < 4; i++) Sv[i + 8] = GetGpsInt(p, 7 + (i * 4));    // third
        if(p[6] == p[8]) for(FoundSatCnt = i = 0; i < 12; i++) if(Sv[i] > 3) FoundSatCnt++;

	// decode the RMC message (this gives us just speed and heading)
    } else if(p[2] == 'R' && p[3] == 'M' && p[4] == 'C') {
        if(p[17] != 'A') {
            NoSignal = true;                                        // we don't have a good lock on enough satellites
        } else {
            Startup = NoSignal = false;                             // good signal so reset the signal related flags

            Speed = GetGpsInt(p, 7);            					// get the GPS speed (in Knots)
            if(Speed > 0 && GetGpsDecimal(p, 7) > 50) Speed++;		// round up if a fraction of a knot
            if(Speed > 0 && eeprom_read(E_speedunits) == 0) 
            	Speed = GetGpsInt(p, 7) + ((GetGpsInt(p, 7)*85)+(GetGpsDecimal(p, 7)*2)+50)/100;	// convert to kmh
            if(eeprom_read(E_speedunits) == 2) Speed = ((uint16)Speed * 115)/100;					// convert to mph
            
            // set the overspeed alarm flag, includes hysterisis of 3 speed units.  Don't set alarm if we are setting the alarm
            if(Speed > eeprom_read(E_ospeed)) OverSpeed = true;
            if(Speed < eeprom_read(E_ospeed) - 2) OverSpeed = false;
            if(eeprom_read(E_ospeed) == 0 || State == S_SpeedCfg) OverSpeed = false;

            if(Speed > 0) {
                // get the heading from the gps and round up based on the fraction of a degree
                Heading = GetGpsInt(p, 8);
                if(GetGpsDecimal(p, 8) > 50) Heading++;
            }
        }
    }
    
    GpsDataReady = 0;                                               // signal that we have processed the data
}



// Get the integer value of a field in a gps message
// p = pointer to the text buffer
// i = field number.  the first field after the $GPXXX is 1, the next 2, etc.
// this will stop at the first non numeric character and return a 16 bit unsigned integer
uint16 GetGpsInt(uint8 *p, uint8 i) {
    uint16 t;
    
    while(i) if(*p++ == ',') i--;
    t = 0;
    while(*p >= '0' && *p <= '9') t = (t * 10) + (*p++ - '0');  
    return t;
}   


// Get the value after the decimal point in a field in a gps message
// p = pointer to the text buffer
// i = field number.  the first field after the $GPXXX is 1, the next 2, etc.
// this will stop at the first non numeric character and return a 16 bit unsigned integer
// representing the decimal value * 100 (ie, 2 digits)
// will return zero if no decimal point found
uint16 GetGpsDecimal(uint8 *p, uint8 i) {
    uint16 t;
    
    while(i) if(*p++ == ',') i--;
    while(*p != '.') {
        if(*p == ',') return 0;                                 // find the decimal point
        p++;
    }   
    t = 0;
    p++;
    if(*p >= '0' && *p <= '9') {
        t = (*p++ - '0') * 10;
        if(*p >= '0' && *p <= '9') t += (*p++ - '0');
    }   
    return t;
}   


// get the time from a gps message
void GetTime(uint8 *p) {
    //if(p[12] != '.' || p[16] != ',') return;
    hour = ((p[6] - '0') * 10) + (p[7] - '0');
    min = ((p[8] - '0') * 10) + (p[9] - '0');
    sec = ((p[10] - '0') * 10) + (p[11] - '0');
}   


// get latitude or longtitude from a gps message
void GetLatLong(uint8 *p, uint8 *b) {
    uint8 i;
    
    //if(p[4] != '.' || p[9] != ',' || p[11] != ',') return;
    *b++ = *p++; *b++ = *p++;
    *b++ = '['; *b++ = ' ';                         // note that [ is the degree symbol
    for(i = 0; i < 7; i++) *b++ = *p++;
    *b++ = '\''; *b++ = ' ', *b++ = *++p;
    *b = 0;
}   



/************************************************************************************************
State Machine functions
The following functions are responsible for handling a specific screen on the LCD
They are called by the main state machine depending on the current state
Each function is responsible for checking the data available and button pressed flags and doing
what ever is necessary.
If there is nothing to do they should simply return.
The various flags should not be reset inside these functions.  It is the responsibility of the state
machine to reset them.
Each function called by the state machine has a name in the format of DoXxx where Xxx is the screen.
************************************************************************************************/

void DoKmRemaining(void) {
    static sint32 LastTime = -1;
    static uint8 SecCnt = 0;
    static uint16 SpeedMin = 0;
    sint32 ThisTime;
    uint8 i;
    uint16 t;
    
    ThisTime = (sint32)hour * 3600 + (sint32)min * 60 + (sint32)sec;
    if(ThisTime == LastTime) return;
    if(((ThisTime - LastTime) > 1) || Speed == 0 || State == S_DistCfg) { SecCnt = 0; SpeedMin = 0; LastTime = ThisTime; return; }
    
    KmSecRemaining -= Speed;
    if(KmSecRemaining < 0) KmSecRemaining = 0;
    t = KmSecRemaining/3600;
    if(WordLoByte(t) != eeprom_read(E_dist_lo)) {
        eeprom_write(E_dist_hi, WordHiByte(t));
        eeprom_write(E_dist_lo, WordLoByte(t));
    }
    
    SpeedMin += Speed;
    if(++SecCnt >= 60) {
        for(i = 1; i < KM_MIN_SIZE; i++)
            KmMin[i - 1] = KmMin[i];
        KmMin[KM_MIN_SIZE - 1] = SpeedMin;
        SecCnt = 0;
        SpeedMin = 0;
    }
    LastTime = ThisTime;
}   



// convert the time from UTC into local time using the timezone
void CalcTime(void) {
    minutes = (sint16)hour * 60 + (sint16)min + (sint16)tz * 6;
    if(minutes < 0) minutes += 24 * 60;
    if(minutes > 24 * 60) minutes -= 24 * 60;
    h = minutes/60;
    if(h >= 12) { 
        pm = true; h -= 12;
    } else
        pm = false;
    if(h == 0) h = 12;
    m = minutes % 60;
}


void DoTime(void) {
    if(NewData) {
        glcd_fillScreen(0);
        glcd_rect(43,7,46,13,true,ON); glcd_rect(43,19,46,25,true,ON);                  // draw the colon
        glcd_setxy(-5, 2, 1); printf(glcd_putc2232, "%2u", h);
        glcd_setxy(49, 2, 1);   printf(glcd_putc2232, "%02u", m);
        glcd_setxy(102, 1, 1); 
        if(pm)
            glcd_putc812('P');
        else
            glcd_putc812('A');
        glcd_putc812('M');
        glcd_setxy(98, 16, 1); printf(glcd_putc1116, "%02u", sec);
        glcd_update();
    }   
}   




void DoTimeCfg(void) {
    if(UpBtn) {
        while(UpBtn) { UpBtn--; tz += 5; if(tz > 120) tz = -120; }
         eeprom_write(E_tz, tz); 
    }
    else if(DwnBtn) {
        while(DwnBtn) { DwnBtn--; tz -= 5; if(tz < -120) tz = 120; }
         eeprom_write(E_tz, tz);
    }
    
    if(NewData) {
        glcd_fillScreen(0);
        CalcTime();
        strcpy(StrBuf, M_settime); Center812(StrBuf, 60, 0);            // print set time heading       
        sprintf(StrBuf, "%u:%02u %cM", h, m, pm?'P':'A');
        Center1116(StrBuf, 60, 17);
        glcd_update();
    }   
}   



void DoSpeed(void) {
    bit Rev;
    if(NewData) {
        Rev = ((Speed > eeprom_read(E_ospeed)) && (eeprom_read(E_ospeed) != 0));
        glcd_fillScreen(Rev);
        *StrBuf = 0; CopySpeedUnits();													// get the speed units
        glcd_colour = !Rev; Center1116(StrBuf, 104, 1);									// display speed units
        if(eeprom_read(E_ospeed)) { 
            glcd_setxy(85, 20, 1, !Rev); 
            printf(glcd_putc812, "+%u", eeprom_read(E_ospeed));                         // display speed alarm threshold
        }
        glcd_setxy(2, 2, 1, !Rev); printf(glcd_putc2232, "%3u", Speed);                 // display the speed
        glcd_update();
        glcd_colour = 0;
    }   
}   


void CopySpeedUnits(void){
	uint8 *t = StrBuf;
	
	while(*t) t++;
	if(eeprom_read(E_speedunits) == 0)
		strcpy(t, M_kmh);
	else
		if(eeprom_read(E_speedunits) == 1)
			strcpy(t, M_kn);
		else
			strcpy(t, M_mph);
}



void CopyDistanceUnits(void){
	uint8 *t = StrBuf;
	
	while(*t) t++;
	if(eeprom_read(E_speedunits) == 0)
		strcpy(t, M_km);
	else
		if(eeprom_read(E_speedunits) == 1)
			strcpy(t, M_nm);
		else
			strcpy(t, M_miles);
}



void DoSpeedUnitsCfg(void) {
    if(UpBtn || DwnBtn) {
	    sint8 t;
	    t = eeprom_read(E_speedunits);
        if(UpBtn)
        	t++;
        else
        	t--;
        if(t > 2) t = 0;
        if(t < 0) t = 2;
        eeprom_write(E_speedunits, t);
    }
    
    glcd_fillScreen(0); 
    strcpy(StrBuf, M_setspeedunits); Center812(StrBuf, 60, 0);         // print the heading    
	*StrBuf = 0; CopySpeedUnits();   
    Center1116(StrBuf, 60, 16);                                   // print speed units
    glcd_update();  
}   




void DoSpeedCfg(void) {
    uint8 ospeed;
    ospeed = eeprom_read(E_ospeed);
    
    if(UpBtn) {
        while(UpBtn) { UpBtn--; if(ospeed != 250) ospeed++; }
        eeprom_write(E_ospeed, ospeed);
    }   
    else if(DwnBtn) {
        while(DwnBtn) { DwnBtn--; if(ospeed != 0) ospeed--; }
        eeprom_write(E_ospeed, ospeed);
    }   

    if(NewData) {
        glcd_fillScreen(0); 
        strcpy(StrBuf, M_setspeed); Center812(StrBuf, 60, 0);                           // print set ospeed heading 
        glcd_setxy(0, 16, 1);  
        if(eeprom_read(E_ospeed)) {
            sprintf(StrBuf, "%u", ospeed); 
            CopySpeedUnits();
        }       
        else
            strcpy(StrBuf, M_off);
            
        Center1116(StrBuf, 60, 16);
        glcd_update();
    }   
}   



void DoFuel(void){
    uint8 i;

    if(NewData) {
        glcd_fillScreen(0);
        for(i = 1; i <= 121; i += 24) glcd_line(i, 0, i, 31, ON);
        for(i = 13; i < 120; i += 24) glcd_line(i, 4, i, 27, ON);
        glcd_rect(1, 8, 121, 23, ON, OFF);
        glcd_rect(1, 8, 121, 23, OFF, ON);
        glcd_rect(1, 8, FuelEfficiency + 1, 23, ON, ON);
        //sprintf(StrBuf, "%Ld", FuelCnt2);
        //Center1116(StrBuf, 60, 16);
        glcd_update();
    }
}




void DoFuelCfg(void){
    uint8 i;
    i = eeprom_read(E_fuel);
    
    if(UpBtn) {
        while(UpBtn) { UpBtn--; if(i < 255) i++; }
        eeprom_write(E_fuel, i); 
    }
    else if(DwnBtn) {
        while(DwnBtn) { DwnBtn--; if(i > 1) i--; }
        eeprom_write(E_fuel, i); 
    }
    
    if(NewData) {
        glcd_fillScreen(0); 
        strcpy(StrBuf, M_setfuel); Center812(StrBuf, 60, 0);                            // print the heading    
        sprintf(StrBuf, "%u", i);
        Center1116(StrBuf, 60, 16);
        glcd_update();
    }   
}



void DoDist(void){
    uint8 i, j;
    sint32 t;

    if(NewData) {
        glcd_fillScreen(0);
        sprintf(StrBuf, "%Ld", (sint16)(KmSecRemaining/3600));
        CopyDistanceUnits();
        Center1116(StrBuf, 60, 0);
        t = 0;
        for(j = i = 0; i < KM_MIN_SIZE; i++)
            if(KmMin[i] > 1200) { 
                j++; t += KmMin[i];
            }   
        if(j > 0 && KmSecRemaining > 0) {
            t = (KmSecRemaining * j) / t ;
            sprintf(StrBuf, "%Ldh %dm", (sint16)(t/60), (sint8)(t%60));
            Center1116(StrBuf, 60, 16);
        } else {
            strcpy(StrBuf, M_remain); 
            Center812(StrBuf, 60, 17);
        }   
        glcd_update();
    }
}




void DoDistCfg(void){
    uint16 t;
    
    if(UpBtn || DwnBtn) {
        if(UpBtn)
            while(UpBtn--) 
                KmSecRemaining += 3600; 
        else    // DwnBtn
            while(DwnBtn--) 
                KmSecRemaining -= 3600;

        if(KmSecRemaining < 0) 
            KmSecRemaining = 0;
        else
            { KmSecRemaining /= 3600; KmSecRemaining *= 3600; KmSecRemaining += 3600/4; }
        t = KmSecRemaining / 3600;
        eeprom_write(E_dist_hi, WordHiByte(t));
        eeprom_write(E_dist_lo, WordLoByte(t)); 
    }
    
    if(NewData) {
        glcd_fillScreen(0); 
        strcpy(StrBuf, M_setdest); Center812(StrBuf, 60, 0);                            // print the heading    
        if(KmSecRemaining) {
            sprintf(StrBuf, "%Ld", KmSecRemaining/3600);
            CopyDistanceUnits();
		}
        else
            strcpy(StrBuf, M_off);  
        Center1116(StrBuf, 60, 16);
        glcd_update();
    }   
}


// sine function lookup table.  Each step is one degree starting from zero and going up to 89 degrees.
const uint8 lu[90] = {0, 4, 8, 13, 17, 22, 26, 31, 35, 40, 44, 48, 53, 57, 61, 66, 70, 74, 79, 83, 87, 91, 95, 100, 104, 108, 112, 116,
					  120, 124, 128, 131, 135, 139, 143, 146, 150, 154, 157, 161, 164, 167, 171, 174, 177, 181, 184, 187, 190, 193, 196,
                      198, 201, 204, 207, 209, 212, 214, 217, 219, 221, 223, 226, 228, 230, 232, 233, 235, 237, 238, 240, 242, 243, 244,
                      246, 247, 248, 249, 250, 251, 252, 252, 253, 254, 254, 255, 255, 255, 255, 255};


// get the x coordinate (horizontal displacement) given a compass reading
// d is the desired direction of the pointer in degrees
// c is the horizontal center of the arrow in pixels
uint8 GetX(uint16 d, uint8 c) {
    if(d < 90) return c + (lu[d] >> 4);
    if(d < 180) return c + (lu[89 - ((d-90))] >> 4);
    if(d < 270) return c - (lu[d-180] >> 4);
    return c - (lu[89 - (d-270)] >> 4); 
}       
    
    
// get the y coordinate (vertical displacement) given a compass reading
// d is the desired direction of the pointer in degrees
// c is the vertical center of the arrow in pixels
uint8 GetY(uint16 d, uint8 c) {
    if(d < 90) return c - (lu[89 - d] >> 4);
    if(d < 180) return c + (lu[d-90] >> 4);
    if(d < 270) return c + (lu[89 - (d-180)] >> 4);
    return c - (lu[d-270] >> 4); 
}       



void DoAlt(void){
    #define HEADING_CENTER  30
    #define NBR_OFFSET      18
    uint8 x1, y1, x2, y2;
    sint16 i, j, k, cp, CompassPoint;
    
    if(NewData) {
        glcd_fillScreen(0);
        if(eeprom_read(E_pointer))
            CompassPoint = Heading;
        else
            CompassPoint = 360 - Heading;
        if(CompassPoint == 360) CompassPoint = 359;

		for(k = -2; k <= 2; k++) {
	        cp = CompassPoint + k;
	        if(cp >= 360) cp -= 360;
	        if(cp < 0) cp += 360;
	        for(i = - 16; i <= 16; i += 1) {
	            j = cp + i + 180;
	            if(j < 0) j += 360;
	            if(j >= 360) j -= 360;
	            x1 = GetX(cp, HEADING_CENTER);
	            y1 = GetY(cp, 16);
	            x2 = GetX(j, HEADING_CENTER);
	            y2 = GetY(j, 16);
	            glcd_line(x1, y1, x2, y2, ON);
	        }
	 	}  
        glcd_circle(HEADING_CENTER, 16, 5, ON, ON);
        glcd_circle(HEADING_CENTER, 16, 2, ON, OFF);
        glcd_pixel(HEADING_CENTER, 16, ON);
        if(CompassPoint < 90 || (CompassPoint > 180 && CompassPoint <= 270))
            i = 0;
        else
            i = 21;
        glcd_setxy(0, i, 1); printf(glcd_putc812, "%Ld[", Heading);     // display the heading
        strcpy(StrBuf, M_alt); Center812(StrBuf, 92, 0);                // display ALTITUDE
		if(eeprom_read(E_altitudeunits) == 0)
	        sprintf(StrBuf, "%Ldm", Altitude);
	    else
	        sprintf(StrBuf, "%Ldft", Altitude);
	    Center1116(StrBuf, 92, 17);
        glcd_update();
    }   
}



void DoAltUnitsCfg(void) {
    if(UpBtn || DwnBtn) {
        if(eeprom_read(E_altitudeunits))
            eeprom_write(E_altitudeunits, 0);
        else
            eeprom_write(E_altitudeunits, 1);
    }
    
    glcd_fillScreen(0); 
    strcpy(StrBuf, M_setheightunits); Center812(StrBuf, 60, 0);         // print the heading    
	if(eeprom_read(E_altitudeunits) == 0)
		strcpy(StrBuf, M_metres);
	else
		strcpy(StrBuf, M_feet);
    Center1116(StrBuf, 60, 16);                                   // print speed units
    glcd_update();  
}   



void DoAltCfg(void) {
    if(UpBtn || DwnBtn) {
        if(eeprom_read(E_pointer))
            eeprom_write(E_pointer, 0);
        else
            eeprom_write(E_pointer, 1);
    }
    
    glcd_fillScreen(0); 
    strcpy(StrBuf, M_setpointer); Center812(StrBuf, 60, 0);         // print the heading    
    if(eeprom_read(E_pointer))
        strcpy(StrBuf, M_heading);
    else
        strcpy(StrBuf, M_due_north);    
    Center1116(StrBuf, 60, 16);                                     // print north/heading
    glcd_update();  
}   



void DoPosit(void) {
    if(NewData) {
        glcd_fillScreen(0);
        glcd_setxy(0, 2, 1); glcd_text812(LatBuf);
        glcd_setxy(0, 16, 1); glcd_text812(LongBuf);
        glcd_update();
    }   
}   



void DoSat(void) {
    uint8 i;
    
    #define BAR_START   50
    #define BAR_WIDTH   5
    
    if(NewData) {
        glcd_fillScreen(0);
        for(i = 0; i < 12; i++) {
            if(Sv[i] > 62) Sv[i] = 62;                                  // make sure the value will plot within the LCD size
            glcd_rect(BAR_START + (i * (BAR_WIDTH + 1)), 31 - Sv[i]/2, BAR_START + (i * (BAR_WIDTH + 1)) + BAR_WIDTH - 1, 31, true, ON);
        }
        glcd_setxy(0, 4, 1); printf(glcd_putc812, "%2u SAT", SvCnt);
        if(SatUsed > SvCnt) SatUsed = SvCnt;
        if(SatUsed < 10)
            glcd_setxy(4, 17, 1); 
        else
            glcd_setxy(0, 17, 1); 
        printf(glcd_putc812, "USE %u", SatUsed);
        glcd_update();  
    }   
}   




void DoBrightCfg(uint8 ENbr) {
    uint8 i;
    i = eeprom_read(ENbr);
    
    if(UpBtn) {
        while(UpBtn--) if(i != 100) i++; 
        eeprom_write(ENbr, i);
    }   
    else if(DwnBtn) {
        while(DwnBtn--) if(i != 0) i--; 
        eeprom_write(ENbr, i);
    }   
    
    BkLightValue = i;
    
    if(NewData) {
        glcd_fillScreen(0); 
        Center812(StrBuf, 60, 0);   
        sprintf(StrBuf, "%u%%", i);
        Center1116(StrBuf, 60, 16);
        glcd_update();
    }   
}   




void DoBrightDayCfg(void) {
    strcpy(StrBuf, M_bright_day);
    DoBrightCfg(E_bright_day);  
}   



void DoBrightNightCfg(void) {
    strcpy(StrBuf, M_bright_night);
    DoBrightCfg(E_bright_night);    
}   



/*********************************************************************************************
General purpose or utility functions
*********************************************************************************************/


// Set up the GPS serial interface for 4800 baud async receive and transmit
void GPS_SetupUART(void) {
    SPBRGH = 0;
    SPBRG = 155;                                                    // baud rate selector, 4800 baud
    //BAUDCON = 0b00100000;
    RCIE = OFF;                                                     // no interrupts
    TXEN = ON;                                                      // enable transmit
    BRGH = OFF;                                                     // low speed
    SPEN = ON;                                                      // enable serial port
    CREN = ON;                                                      // enable receive
}

    
        
void Center812(uint8 *p, uint8 x, uint8 y) {
    uint8 j;
    
    for(j = 0; p[j]; j++) x -= 4;
    glcd_setxy(x, y);
    glcd_text812(p);
}




void Center1116(uint8 *p, uint8 x, uint8 y) {
    uint8 j;
    
    for(j = 0; p[j]; j++) x -= 6;
	glcd_x = x; glcd_y = y;         								// Coordinates
	glcd_size = 1;              									// Size
    glcd_text1116(p);
}


// implements the Show-Hide mode
void SetShowHide(void) {
    uint8   st = 1;
    uint8 i, j;
    
    enable_interrupts(GLOBAL);
    glcd_update();

    while(!UP_BTN);
    UpBtnEvent = 0;
    
    while(forever) {
        glcd_fillScreen(0); 
        j = i = 0;
        while(i != st) if(M_set_showlist[j++] == 0) i++;
        for(i = 0; M_set_showlist[j] != 0; i++, j++) StrBuf[i] = M_set_showlist[j];
        StrBuf[i] = 0;
        Center812(StrBuf, 60, 0);                           // print the heading
        
        i = eeprom_read(HIDE_START + st);
        switch(i) {                                         // copy the show/hide string
            case 0: strcpy(StrBuf, M_set_show); break;
            case 1: strcpy(StrBuf, M_set_show_auto); break;
            case 2: strcpy(StrBuf, M_set_hide); break;
        }
        Center812(StrBuf, 60, 17);                          // print the show/hide string
        glcd_update();
        
        if(SetBtnEvent) {
            if(i++ == 2) i = 0;
            eeprom_write(HIDE_START + st, i);
        }   
        if(DwnBtnEvent) if(++st == HIDE_SIZE) st = 0;
        if(UpBtnEvent) if(st-- == 0) st = HIDE_SIZE - 1;
        SetBtnEvent = UpBtnEvent = DwnBtnEvent = 0;
    }
}


// reset everything to "factory defaults"
void ResetAll(void) {
    uint8 i;
    
    mSec(500);
    
    // send the reset to factory defaults command to the GPS module
    for(i = 0; M_gps_reset[i]; i++) {
        while(!TXIF);
        TXREG = M_gps_reset[i];
    }
    // copy the backup default settings into the working area of the eeprom
    for(i = 0; i < HIDE_START + HIDE_SIZE; i++)
        eeprom_write(i, eeprom_read(BACKUP_START + i));
}   


